# -*- coding: utf-8 -*-
"""
Created on Wed Dec 17 01:11:47 2025

@author: mrwol
"""

# -*- coding: utf-8 -*-
import pandas as pd
import plotly.graph_objects as go
from plotly.colors import sample_colorscale
from dash import Dash, dcc, html, Input, Output, State
import dash_bootstrap_components as dbc
from dash.exceptions import PreventUpdate
import base64
import io
import numpy as np

# --------------------- Глобальные переменные ---------------------
df = pd.DataFrame()

# --------------------- Палитры ---------------------
PALETTE_PRESETS = {
    "Business (спокойный)": [
        'Blues', 'Greys', 'YlGnBu', 'Cividis'
    ],
    "Tech Style (яркий)": [
        'Viridis', 'Cividis', 'Turbo', 'Plasma'
    ],
    "Analytics (контрастный)": [
        'Viridis', 'Cividis', 'YlGnBu', 'Blues'
    ],
    "Dark (ночной)": [
        'Inferno', 'Magma', 'Plasma', 'Viridis'
    ],
    "Soft (пастель)": [
        'Teal', 'YlGn', 'Greens', 'Blues'
    ]
}

# --------------------- Dash ---------------------
# Используем светлую тему по умолчанию
app = Dash(__name__, external_stylesheets=[dbc.themes.BOOTSTRAP])
app.title = "Интерактивная древовидная диаграмма"
app.config.suppress_callback_exceptions = True  # для динамических элементов

# --------------------- Layout ---------------------
# Базовый стиль для dropdown, чтобы избежать проблем с цветом текста при первом рендере
INITIAL_DROPDOWN_STYLE = {'color': 'black', 'backgroundColor': 'white'}

app.layout = dbc.Container([
    dcc.Store(id="theme-store", data="light"), # Начальная тема "light"

    # Заголовок и переключатель темы в одном ряду
    dbc.Row([
        dbc.Col(html.H2("Интерактивная древовидная диаграмма", className="mt-3", id="app-title"), width=9, className="d-flex align-items-center"),
        dbc.Col(dbc.Switch(id="theme-switch", label="Тёмная тема", value=False), width=3, className="d-flex align-items-end justify-content-end mt-3")
    ], className="mb-3"),

    # Блок управления в виде карточки для компактности
    dbc.Card(
        [
            # 1. Загрузка файла
            dbc.Row([
                dbc.Col([
                    html.Label("Загрузка данных (CSV/Excel)", id="upload-label", className="mb-1"),
                    dcc.Upload(
                        id='upload-data',
                        children=html.Div(['Перетащите файл или нажмите для выбора'], className="p-2"),
                        style={
                            'borderWidth': '1px', 'borderStyle': 'dashed',
                            'borderRadius': '5px', 'textAlign': 'center',
                            'cursor': 'pointer', 'minHeight': '50px',
                            'backgroundColor': 'rgba(255, 255, 255, 0.05)', # Стиль для светлой темы (обновится колбэком)
                            'borderColor': '#ccc'
                        },
                        multiple=False
                    ),
                    html.Div(id='uploaded-filename', className='mt-2 mb-0 small')
                ], md=4, className="pr-md-4"), # Колонка для загрузки

                # 2. Выбор иерархии и значений
                dbc.Col([
                    dbc.Row([
                        dbc.Col([html.Label("Уровни иерархии", id="level-label", className="mb-1"), dcc.Dropdown(id='level-dropdown', multi=True, placeholder="Выберите колонки", style=INITIAL_DROPDOWN_STYLE)], md=6),
                        dbc.Col([html.Label("Колонка значений", id="value-label", className="mb-1"), dcc.Dropdown(id='value-dropdown', placeholder="Выберите колонку", style=INITIAL_DROPDOWN_STYLE)], md=6),
                    ], className="mb-3"),

                    # 3. Агрегация, Тип диаграммы и Цветовой пресет
                    dbc.Row([
                        dbc.Col([
                            html.Label("Тип диаграммы", id="chart-label", className="mb-1"),
                            dcc.Dropdown(
                                id='chart-type',
                                options=[{'label': 'Sunburst', 'value': 'Sunburst'}, {'label': 'Treemap', 'value': 'Treemap'}, {'label': 'Icicle', 'value': 'Icicle'}],
                                value='Treemap',
                                clearable=False,
                                style=INITIAL_DROPDOWN_STYLE
                            )
                        ], md=4),
                        dbc.Col([
                            html.Label("Агрегация", id="agg-label", className="mb-1"),
                            dcc.Dropdown(
                                id='agg-type',
                                options=[{'label': 'SUM', 'value': 'SUM'}, {'label': 'COUNT', 'value': 'COUNT'}, {'label': 'AVG', 'value': 'AVG'}],
                                value='SUM',
                                clearable=False,
                                style=INITIAL_DROPDOWN_STYLE
                            )
                        ], md=4),
                        dbc.Col([
                            html.Label("Цветовой пресет", id="palette-label", className="mb-1"),
                            dcc.Dropdown(
                                id='palette-preset',
                                options=[{'label': k, 'value': k} for k in PALETTE_PRESETS.keys()],
                                value='Business (спокойный)',
                                clearable=False,
                                style=INITIAL_DROPDOWN_STYLE
                            )
                        ], md=4),
                    ]),
                ], md=8),
            ], align="start") # Выравнивание по верхнему краю
        ],
        body=True,
        className="shadow-sm mb-4", # Добавляем тень и отступ
        id="controls-card"
    ),

    # График
    dcc.Graph(id='graph', style={'height': '80vh'}),

    html.Div(id="dummy", style={"display": "none"})
], fluid=True, id='main-container')


# --------------------- Callback для темы (только сохраняет состояние) ---------------------
@app.callback(
    Output("theme-store", "data"),
    Input("theme-switch", "value")
)
def toggle_theme(dark):
    return "dark" if dark else "light"

# --------------------- CALLBACK для обновления темы ВСЕГО ПРИЛОЖЕНИЯ ---------------------
@app.callback(
    Output('main-container', 'style'),
    Output('app-title', 'className'),
    Output('controls-card', 'className'),
    Output('upload-data', 'style'),
    Output('uploaded-filename', 'style'),
    # Обновление стилей для меток (Label) в карточке
    [Output(f'{id}', 'style') for id in ['upload-label', 'level-label', 'value-label', 'chart-label', 'agg-label', 'palette-label']],
    # НОВЫЕ ВЫХОДЫ: Стили для dcc.Dropdown
    Output('level-dropdown', 'style'),
    Output('value-dropdown', 'style'),
    Output('chart-type', 'style'),
    Output('agg-type', 'style'),
    Output('palette-preset', 'style'),
    
    Input('theme-store', 'data'),
    State('upload-data', 'style'),
)
def update_app_theme(theme, upload_style_state):
    # Определение общих стилей
    if theme == "dark":
        bg_color = "#2c2c2c"
        text_color = "white"
        card_bg = "#363636" # Чуть светлее фона для контраста карточки
        
        container_style = {"backgroundColor": bg_color, "color": text_color}
        title_class = "mt-3 text-white"
        label_style = {'color': text_color, 'marginBottom': '0.25rem'}
        
        # Используем bg-secondary (Bootstrap class) или просто text-white
        card_class = "shadow-lg mb-4 text-white bg-secondary" 
        
        # Обновление стиля для Upload-блока
        upload_style = {
            'borderWidth': '1px', 'borderStyle': 'dashed',
            'borderRadius': '5px', 'textAlign': 'center',
            'cursor': 'pointer', 'minHeight': '50px',
            'backgroundColor': '#3f3f3f',
            'borderColor': '#555',
            'color': text_color
        }
        filename_style = {'color': 'lightgray', 'marginTop': '0.5rem', 'fontWeight': 'bold'}
        
        # Стиль для выпадающих списков
        dropdown_style = {
            'backgroundColor': '#3f3f3f', # Темный фон для поля ввода
            'color': '#4296F5' # Синий текст в выпадающих списках
        }

    else:
        bg_color = "white"
        text_color = "black"
        
        container_style = {"backgroundColor": bg_color, "color": text_color}
        title_class = "mt-3 text-black"
        label_style = {'color': text_color, 'marginBottom': '0.25rem'}
        card_class = "shadow-sm mb-4 bg-white"
        
        upload_style = {
            'borderWidth': '1px', 'borderStyle': 'dashed',
            'borderRadius': '5px', 'textAlign': 'center',
            'cursor': 'pointer', 'minHeight': '50px',
            'backgroundColor': 'white',
            'borderColor': '#ccc',
            'color': text_color
        }
        filename_style = {'color': 'black', 'marginTop': '0.5rem', 'fontWeight': 'bold'}
        
        dropdown_style = {
            'backgroundColor': 'white',
            'color': 'black'
        }


    label_styles = [label_style] * 6 # Применяем одинаковый стиль ко всем меткам
    dropdown_styles = [dropdown_style] * 5
    
    return (
        container_style,
        title_class,
        card_class,
        upload_style,
        filename_style,
        *label_styles,
        *dropdown_styles
    )


# --------------------- Callback для загрузки данных ---------------------
@app.callback(
    Output('level-dropdown', 'options'),
    Output('value-dropdown', 'options'),
    Output('uploaded-filename', 'children'),
    Input('upload-data', 'contents'),
    State('upload-data', 'filename')
)
def update_columns(contents, filename):
    if contents is None:
        raise PreventUpdate

    content_type, content_string = contents.split(',')
    decoded = base64.b64decode(content_string)

    global df
    try:
        if filename.endswith('.csv'):
            df = pd.read_csv(io.StringIO(decoded.decode('utf-8')))
        elif filename.endswith('.xls') or filename.endswith('.xlsx'):
            df = pd.read_excel(io.BytesIO(decoded))
        else:
            return [], [], f"Неподдерживаемый формат файла: {filename}"
    except Exception as e:
        print(f"Ошибка загрузки файла: {e}")
        return [], [], f"Ошибка загрузки файла: {filename}. {e}"

    cols = [{'label': c, 'value': c} for c in df.columns]
    return cols, cols, f"Загружен: {filename}"

# --------------------- Callback для построения графика ---------------------
@app.callback(
    Output('graph', 'figure'),
    Input('level-dropdown', 'value'),
    Input('value-dropdown', 'value'),
    Input('chart-type', 'value'),
    Input('agg-type', 'value'),
    Input('palette-preset', 'value'),
    Input('theme-store', 'data')
)
def update_graph(levels, val, chart, agg, palette_preset, theme):
    
    # 1. Определение цветов и шаблона Plotly
    if theme == "dark":
        bg_color = "#2c2c2c"
        text_color = "white"
        template_name = "plotly_dark"
    else:
        bg_color = "white"
        text_color = "black"
        template_name = "plotly_white"
        
    # Начальная проверка и возврат заглушки
    if df.empty or not levels or not val:
        fig = go.Figure()
        
        fig.update_layout(
            title_text="Загрузите файл и выберите параметры",
            xaxis={'visible': False}, yaxis={'visible': False},
            annotations=[dict(text="Нет данных для отображения", x=0.5, y=0.5, showarrow=False, font_size=20, font=dict(color=text_color))],
            plot_bgcolor=bg_color,
            paper_bgcolor=bg_color,
            font=dict(color=text_color)
        )
        return fig

    try:
        df_clean = df.copy()
        if agg in ['SUM', 'AVG']:
            df_clean[val] = pd.to_numeric(df_clean[val], errors='coerce').fillna(0)

        # 2. Агрегация
        if agg == "SUM":
            df_agg = df_clean.groupby(levels, as_index=False)[val].sum()
        elif agg == "AVG":
            df_agg = df_clean.groupby(levels, as_index=False)[val].mean()
        else:  # COUNT
            df_agg = df_clean.groupby(levels, as_index=False).size().rename(columns={"size": val})

        # 3. Цветовые схемы
        preset_palettes = PALETTE_PRESETS.get(palette_preset, list(PALETTE_PRESETS.values())[0])
        level_colors = {lvl: preset_palettes[i % len(preset_palettes)] for i, lvl in enumerate(levels)}

        # 4. Рекурсивное создание узлов
        nodes = []
        node_colors = {}
        node_id_counter = 0

        def normalize(x):
            if len(x) == 0: return np.array([])
            mn, mx = x.min(), x.max()
            if mn == mx: return np.zeros(len(x))
            return (x - mn) / (mx - mn)

        def add_nodes(level_idx, parent_vals=None, parent_id=None):
            nonlocal node_id_counter
            lvl = levels[level_idx]

            if parent_vals:
                mask = np.ones(len(df_agg), dtype=bool)
                for k, v in parent_vals.items():
                    mask &= (df_agg[k] == v)
                df_sub = df_agg[mask]
            else:
                df_sub = df_agg

            if df_sub.empty:
                return

            vals_for_color = df_sub[val].values
            norm_vals = normalize(vals_for_color)
            unique_cats = df_sub[lvl].unique()

            for cat in unique_cats:
                rows_mask = df_sub[lvl] == cat
                node_value = df_sub[rows_mask][val].sum() 
                if node_value <= 0: node_value = 1e-9

                cat_norm_val = norm_vals[rows_mask].mean() if len(norm_vals[rows_mask]) > 0 else 0
                cat_norm_val = max(0.0, min(1.0, cat_norm_val))
                
                scale = level_colors[lvl]
                color = sample_colorscale(scale, [cat_norm_val])[0]

                current_id = f"{lvl}_{node_id_counter}"
                node_id_counter += 1
                node_colors[current_id] = color
                parent_node_id = parent_id if parent_id else ""

                nodes.append({
                    'id': current_id,
                    'label': str(cat),
                    'parent': parent_node_id,
                    'value': node_value
                })

                if level_idx + 1 < len(levels):
                    new_parent_vals = parent_vals.copy() if parent_vals else {}
                    new_parent_vals[lvl] = cat
                    add_nodes(level_idx + 1, new_parent_vals, current_id)

        # Запуск построения
        add_nodes(0)

        nodes_df = pd.DataFrame(nodes)
        
        if nodes_df.empty:
            fig = go.Figure()
            fig.update_layout(title_text="Нет данных после обработки")
            return fig

        # 5. Построение графика
        common_args = dict(
            ids=nodes_df['id'],
            labels=nodes_df['label'],
            parents=nodes_df['parent'],
            values=nodes_df['value'],
            branchvalues='total',
            marker=dict(colors=[node_colors[i] for i in nodes_df['id']], line=dict(width=1, color='white')),
            hovertemplate='<b>%{label}</b><br>Value: %{value}<br>%{percentParent} of parent'
        )
        
        if chart == "Sunburst":
            trace = go.Sunburst(**common_args)
        elif chart == "Treemap":
            trace = go.Treemap(**common_args)
        else: # Icicle
            trace = go.Icicle(**common_args)

        fig = go.Figure(trace)
        fig.update_layout(
            template=template_name,
            margin=dict(t=10, l=10, r=10, b=10),
            # Явно устанавливаем цвет фона Plotly, чтобы он соответствовал контейнеру Dash
            plot_bgcolor=bg_color,
            paper_bgcolor=bg_color,
            font=dict(color=text_color)
        )

        return fig

    except Exception as e:
        print(f"CRITICAL ERROR in update_graph: {e}")
        fig = go.Figure()
        fig.update_layout(
            title_text=f"Ошибка построения: {str(e)}",
            xaxis={'visible': False}, yaxis={'visible': False}
        )
        return fig

# --------------------- Запуск ---------------------
if __name__ == '__main__':
    # Импорт webbrowser перемещен сюда для чистоты основного кода
    import webbrowser 
    url = "http://127.0.0.1:8050"
    webbrowser.open(url, new=2)
    app.run(debug=False, use_reloader=False, port=8050)